%This script generates Fig. 8, composed of two figures:

%Panel A: A plot of the average quoted spread in the final period, under the
%baseline case and the private values case, both in the simulations and in
%the theoretical benchmark, for different number of AMMs N.

%Panel B: The same plot with the realized spread.

%Pre-allocate a 9*4 matrix with 9 lines corresponding to 9 values of
%N and 4 columns with the averages in the simulations of QS (adv. selection),
%RS (adv. selection), QS (private values), RS (private values).

mat_av = zeros(9,4);

%Pre-allocate a 9*4 matrix with 9 lines corresponding to 9 values of N
%and 4 columns with the standard deviations/sqrt(observations) of QS (adv. selection),
%RS (adv. selection), QS (private values), RS (private values).

mat_sd = zeros(9,4);

%Parameters
sigma = 5;
Delta_v = 4;
vH = 2+(Delta_v)/2;
vL = 2-(Delta_v)/2;
grid_middle = 8;
grid_size= 69;  
tick= 0.1;

%Fill the matrix by loading the data on last episodes for every sigma.

for N = 2:10
% Create the name of the adverse selection dataset to be loaded
    dataname = sprintf('Data/last_episodes_%d.txt', N);
    % Load the data
    last_episodes = readmatrix(dataname);

    %Compute the quoted spread in the last episode T of each experiment k
    %(K x 1 vector)
    QS = min(last_episodes(:,1:N),[],2) - 2;
    %Compute the realized spread in the last episode T of each experiment k. 
    %First we keep only episodes with a trade, remember column 2*N+2 of last_episodes is v_c.
    index = find(min(last_episodes(:,1:N),[],2)<last_episodes(:,2*N+2));
    %Generate matrix with only the episodes in which there is a trade
    trades = last_episodes(index,:);
    %Then we compute the realized spread in these episodes. 
    %Remember column 2*N+1 of last_episodes and trades is \tilde{v}.
    RS = min(trades(:,1:N),[],2)-trades(:,2*N+1);
    %Fill the matrix with the average of QS and RS.
    mat_av(N-1,1) = mean(QS);
    mat_av(N-1,2) = mean(RS);
    %Fill the matrix with 1.96*std. deviation of QS and RS, divided by the
    %square root of the number of observations (95% confidence interval,
    %central limit theorem).
    [n1,~] = size(last_episodes);
    [n2,~] = size(trades);
    mat_sd(N-1,1) = 1.96*std(QS)/sqrt(n1);
    mat_sd(N-1,2) = 1.96*std(RS)/sqrt(n2);

%We repeat the procedure for the private values case.

% Create the name of the private values case dataset to be loaded
    dataname = sprintf('Data/last_episodes_private_%d.txt', N);
    % Load the data
    last_episodes = readmatrix(dataname);
    %Compute the quoted spread in the last episode T of each experiment k
    %(K x 1 vector)
    QS = min(last_episodes(:,1:N),[],2) - 2;
    %Compute the realized spread in the last episode T of each experiment k. 
    %First we keep only episodes with a trade, remember column 2*N+2 of last_episodes is v_c.
    index = find(min(last_episodes(:,1:N),[],2)<last_episodes(:,2*N+2));
    %Generate matrix with only the episodes in which there is a trade
    trades = last_episodes(index,:);
    %Remember column 2*N+1 of last_episodes and trades is \tilde{v}.
    RS = min(trades(:,1:N),[],2)-trades(:,2*N+1);
    %Fill the matrix with the average of QS and RS.
    mat_av(N-1,3) = mean(QS);
    mat_av(N-1,4) = mean(RS);
    %Fill the matrix with 1.96*std. deviation of QS and RS, divided by the
    %square root of the number of observations (95% confidence interval,
    %central limit theorem).
    [n3,~] = size(last_episodes);
    [n4,~] = size(trades);
    mat_sd(N-1,3) = 1.96*std(QS)/sqrt(n3);
    mat_sd(N-1,4) = 1.96*std(RS)/sqrt(n4);
end


%We create a matrix with the theoretical values for each sigma of QS and RS
%both in the adverse selection case and in the private values case. 
%The indexing is the same as in "mat_av".
%Note that due to the discretization N may affect the value of the least
%competitive NE price (with continuous prices the price would not depend on
%N for N >= 2).

mat_theory = zeros(9,4);

%Case with adverse selection.

%For every N, we will compute the theoretical ask price, the quoted
%spread, and the realized spread.

for N=2:10

%We compute the theoretical expected profit associated with price a.
%We have Pi(a) = Pr(v=vH) Pr(vH + l > a) (a-vH) + Pr(v=vL) Pr(vL + l > a) (a-vL)
%We compute Pi(a) for every price in the grid. 
%NE is a matrix with 2*grid_size+1 lines (all prices on the grid), first
%column is the price, second column the associated profit, third column a
%variable equal to 1 if a is a NE price.

NE = zeros(2*grid_size+1,3);

%Compute the profit for each price.
for i=1:2*grid_size+1
    a = (grid_middle - grid_size*tick - tick)+(i*tick);
    NE(i,1) = a;
    profit = 0.5*(1-normcdf(a-vH,0,sigma))*(a-vH) + 0.5*(1-normcdf(a-vL,0,sigma))*(a-vL);
    NE(i,2) = profit;
end

%Check for each price whether it can be a NE, using Lemma 3 in the Online
%Appendix. We start with the second price, all prices below 2.0 can never be a NE
%price anyway.

%We set NE(:,3) to 1. We will replace with a zero each price for which we
%find a profitable deviation (not quoting anything or undercutting).
NE(:,3) = ones(2*grid_size+1,1);
%We know the first price necessarily generates losses.
NE(1,3) = 0;
for i=2:2*grid_size+1
    a = NE(i,1);
%Check that the profit is positive.    
if NE(i,2) < 0
   NE(i,3) = 0;
end

%Check that having 1/N times the profit is better than the full profit at a lower price.    
for j=1:i-1
    if NE(j,2) > (1/N)*NE(i,2)
    NE(i,3) = 0;
    end
end

end

%Find highest NE price.
index = find(NE(:,3)==1,1,'last');
a = (grid_middle - grid_size*tick - tick)+(index*tick);
%Record the QS:
mat_theory(N-1,1) = a-2;
%Record the RS (profit divided by probability of a trade):
mat_theory(N-1,2) = NE(index,2)/(0.5*(1-normcdf(a-vH,0,sigma)) + 0.5*(1-normcdf(a-vL,0,sigma)));
end

%Case with private values.
%The least competitive Nash equilibrium is to have a = 2.1, for any Delta_v. 
%Hence QS = 0.1. Moreover in the absence of adverse selection RS = QS.
mat_theory(:,3) = [0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.1];
mat_theory(:,4) = mat_theory(:,3);


%% MEAN FINAL QUOTED SPREAD
figure();
hold on;

%Adverse selection case
%Average
plot(2:10, mat_av(:,1), 'b-+', 'LineWidth', 1);
%Confidence Interval
ci_qs_b = fill([2:10, flip(2:10)], [(mat_av(:,1)-mat_sd(:,1))', flip((mat_av(:,1)+mat_sd(:,1))')], 'b');
set(ci_qs_b, 'facealpha', 0.3);  % make the confidence interval transparent
%Theoretical Benchmark
plot(2:10,mat_theory(:,1) ,'--o','Color',[0.0,0.0,1.0])


%Private Values
%Average
plot(2:10, mat_av(:,3), 'r-+', 'LineWidth', 1);
%Confidence Interval
ci_qs_p = fill([2:10, flip(2:10)], [(mat_av(:,3)-mat_sd(:,3))', flip((mat_av(:,3)+mat_sd(:,3))')], 'r');
set(ci_qs_p, 'facealpha', 0.3);  % make the confidence interval transparent
%Theoretical Benchmark
plot(2:10,mat_theory(:,3) ,'--o','Color',[1.0,0.0,0.0])

ylabel("Average Quoted Spread");
xlabel('$N$', 'Interpreter', 'latex', 'FontSize', 14); 

legend('Simulations - Adverse Selection', '95\% Confidence interval', 'Nash Equilibrium - Adverse Selection', 'Simulations - No Adverse Selection', '95\% Confidence interval' ,'Nash Equilibrium - No Adverse Selection', 'Location', 'Northwest','Interpreter','Latex');
legend('Box','off');
ylim([0,5.5]);
hold off;

%save png
filename = sprintf('/Figures/Fig_8_a.png');
saveas(gcf, [pwd filename]);

%% MEAN FINAL REALIZED SPREAD
figure();
hold on;

%Adverse selection case
%Average
plot(2:10, mat_av(:,2), 'b-+', 'LineWidth', 1);
%Confidence Interval
ci_rs_b = fill([2:10, flip(2:10)], [(mat_av(:,2)-mat_sd(:,2))', flip((mat_av(:,2)+mat_sd(:,2))')], 'b');
set(ci_rs_b, 'facealpha', 0.3);  % make the confidence interval transparent
%Theoretical Benchmark
plot(2:10,mat_theory(:,2) ,'--o','Color',[0.0,0.0,1.0])


%Private Values
%Average
plot(2:10, mat_av(:,4), 'r-+', 'LineWidth', 1);
%Confidence Interval
ci_rs_p = fill([2:10, flip(2:10)], [(mat_av(:,4)-mat_sd(:,4))', flip((mat_av(:,4)+mat_sd(:,4))')], 'r');
set(ci_rs_p, 'facealpha', 0.3);  % make the confidence interval transparent
%Theoretical Benchmark
plot(2:10,mat_theory(:,4) ,'--o','Color',[1.0,0.0,0.0])

ylabel("Average Realized Spread");
xlabel('$N$', 'Interpreter', 'latex', 'FontSize', 14); 

legend('Simulations - Adverse Selection', '95\% Confidence interval', 'Nash Equilibrium - Adverse Selection', 'Simulations - No Adverse Selection', '95\% Confidence interval' ,'Nash Equilibrium - No Adverse Selection', 'Location', 'Northwest','Interpreter','Latex');
legend('Box','off');
ylim([0,5.5]);
hold off;

%save png
filename = sprintf('/Figures/Fig_8_b.png');
saveas(gcf, [pwd filename]);